<?php

// imports
if (!defined('NO_IMPORTS')) {
	require_once(WCF_DIR.'lib/system/exception/SystemException.class.php');
	require_once(WCF_DIR.'lib/data/DatabaseObject.class.php');
}

/**
 * The AbstractServer class provides basic functionality to access a remote host.
 * 
 * @author	Sven Kutzner
 * @copyright	2010 deixu.net
 * @license	GNU Lesser General Public License <http://opensource.org/licenses/lgpl-license.php>
 * @package net.deixu.wcf.teamspeak
 * @subpackage	data
 * @category 	TeamSpeak Viewer
 */
abstract class AbstractServer extends DatabaseObject {
	
	protected $resource = null;
	protected $protocol = 'tcp';
	protected $errorNumber;
	protected $errorString;
	protected $timeout = 10;
	
	protected $groupPermissions;
	protected $userPermissions;
	
	protected $error = null;
	protected $whiteList = null;
	
	protected static $connections = array();
	
	public function setError(Exception $e) {
		$this->error = $e;
	}
	
	public function errorOccurred() {
		return ($this->error instanceof Exception);
	}
	
	public function getError() {
		return $this->error;
	}
	
	protected static function stringSection($str, $section = 0, $delimiter = ' ') {
		if(strpos($str, $delimiter) === FALSE)
			return $str;
		$str = explode($delimiter, $str);
		if(count($str) < $section)
			return $str;
		return $str[$section];
	}

	public function getInformation() {
		// parse message
		require_once(WCF_DIR.'lib/data/message/bbcode/MessageParser.class.php');
		$parser = MessageParser::getInstance();
		$parser->setOutputType('text/html');
		return $parser->parse($this->serverInformation, true, false, true, false);
	}

	public function getName() {
		return $this->name;
	}
	
	public function getInfo($value = null) {
		if(!is_null($value) && isset($this->info[$value])) {
			return $this->info[$value];
		}
		return $this->info;
	}
	
	public function getChannels() {
		return $this->channels;
	}

	public function checkWhiteList($channelID) {
		if(is_null($this->whiteList)) $this->whiteList = $this->getPermission('whiteList');
		if(in_array(-1, $this->whiteList) || in_array($channelID, $this->whiteList)) return true;
		return false;
	}
	
	public function getChannel($channelID) {
		if(isset($this->channels[$channelID]) && $this->checkWhiteList($channelID))
			return $this->channels[$channelID];
		return false;
	}

	public function getChannelTree($parentID = 0) {
		if(is_null($parentID)) return $this->channelTree;
		if(isset($this->channelTree[$parentID]))
			return $this->channelTree[$parentID];
		return array();
	}

	public function channelHasClients($channelID) {
		return isset($this->clientsToChannels[$channelID]);
	}

	public function countChannels() {
		return count($this->channels);
	}
	
	public function getClients() {
		return $this->clients;
	}
	
	public function getClient($clientID) {
		if(isset($this->clients[$clientID]))
			return $this->clients[$clientID];
		return false;
	}
	
	public function countClients() {
		return count($this->clients);
	}
	
	public function getClientsInChannel($channelID) {
		$clients = array();
		if($this->channelHasClients($channelID)) {
			foreach($this->clientsToChannels[$channelID] as $clientID) {
				$clients[] = $this->getClient($clientID);
			}
		}
		return $clients;
	}
	
	public function checkImage($file) {
		return false;
	}
	
	
	/**
	 * Initializes a new Server object.
	 * 
	 * @param 	string		$host
	 * @param 	string		$port
	 * @param 	integer		$timeout
	 */
	public function __construct($row = null, $cacheObject = null) {
		if(!is_null($row)) parent::__construct($row);
		if(!is_null($cacheObject)) parent::__construct($cacheObject->data);
	}

	abstract public function readData();
	
	public function handleData($data) {
		if(isset($data['groupPermissions'])) {
			$this->groupPermissions = $data['groupPermissions'];
			unset($data['groupPermissions']);
		}
		if(isset($data['userPermissions'])) {
			$this->userPermissions = $data['userPermissions'];
			unset($data['userPermissions']);
		}
		parent::handleData($data);
	}
	
	public function getPermission($permission = 'canView') {
		$groupIDs = WCF::getUser()->getGroupIDs();
		$userID = WCF::getUser()->userID;
		$result = null;
		if($permission == 'whiteList') $result = array();

		if(isset($this->userPermissions[$userID])) {
			if(is_array($result)) {
				$result = array_merge($result, array_diff($this->userPermissions[$userID][$permission], $result));
			} else {
				if($this->userPermissions[$userID][$permission] != -1)
					$result = $this->userPermissions[$userID][$permission];
			}
		}
		
		foreach($groupIDs as $groupID) {
			if(!isset($this->groupPermissions[$groupID])) continue;
			if(is_array($this->groupPermissions[$groupID][$permission])) {
				$result = array_merge($result, array_diff($this->groupPermissions[$groupID][$permission], $result));
				continue;
			} elseif($this->groupPermissions[$groupID][$permission] == -1) continue;
		
			if(is_null($result)) $result = $this->groupPermissions[$groupID][$permission];
			else $result = $result || $this->groupPermissions[$groupID][$permission]; 
		}
		
		if($permission == 'whiteList') {
			if(is_array($result) && count($result)==0) $result = array(-1);
			return $result;
		}
		
		if(is_null($result)) $result = true;
		
		return (boolean)$result;
	}
	
	public function getTemplate($postfix = '') {
		return $this->templateName.ucfirst($postfix);
	}
	
	public function getIcon() {
		return $this->icon;
	}

	public function getStyle() {
		return $this->style;
	}
	
	/**
	 * Opens a new connection to a remote host.
	 **/ 
	public function connect() {
		if(!is_null($this->resource)) return;
		$this->resource = @stream_socket_client($this->protocol.'://'.$this->queryHost.':'.$this->queryPort, $this->errorNumber, $this->errorString, $this->timeout);
		if ($this->resource === false) {
			throw new SystemException('Can not connect to ' . $this->queryHost . ': '.$this->errorString, 10000);
		}
		if(@stream_set_timeout($this->resource, $this->timeout) === false)
			throw new SystemException('Can not connect to ' . $this->queryHost . ': '.$this->errorString, 10001);
	}
	
	public function disconnect() {
		if(is_null($this->resource)) return;
		@fclose($this->resource);
		$this->resource = null;
	}
	
	public function read($length = 4096) {
		if(is_null($this->resource)) $this->connect();
		
		$data = stream_get_contents($this->resource, $length);
		return trim($data);
	}
	
	public function readLine() {
		if(is_null($this->resource)) $this->connect();
				
		$data = fgets($this->resource);
		return trim($data);
	}
	
	public function send($data) {
		if(is_null($this->resource)) $this->connect();
				
		@stream_socket_sendto($this->resource, $data);
	}
	
	public function sendLine($data, $lineFeed = "\n") {
		$this->send($data.$lineFeed);
	}
}