<?php

require_once(WCF_DIR.'lib/data/server/AbstractServer.class.php');

/**
 * The TeamSpeak3Server class provides access to a TeamSpeak 3 Server.
 * 
 * @author	Sven Kutzner
 * @copyright	2010-2011 deixu.net
 * @license	GNU Lesser General Public License <http://opensource.org/licenses/lgpl-license.php>
 * @package net.deixu.wcf.teamspeak
 * @subpackage	data
 * @category 	TeamSpeak Viewer
 */
class TeamSpeak3Server extends AbstractServer {
	
	const SPACER_ALIGN_LEFT = 'left';
	const SPACER_ALIGN_CENTER = 'center';
	const SPACER_ALIGN_RIGHT = 'right';
	const SPACER_ALIGN_REPEAT = 'repeat';	
	
	protected static $parameterSearch =  array('\\'  ,'/' ,' ' ,'|' ,"\a","\b","\f","\n","\r","\t","\v");
	protected static $parameterReplace = array('\\\\','\/','\s','\p','\a','\b','\f','\n','\r','\t','\v');
	
	public $lastUpdate = 0;
	
	protected $info;
	protected $channels;
	protected $channelTree;
	protected $clients;
	protected $clientsToChannels;
	protected $serverQueryClients;
	protected $serverGroups;
	protected $channelGroups;
	
	public function connect() {
		parent::connect();
		if($this->readLine() != 'TS3') {
			throw new SystemException('This is not a TeamSpeak 3 Server.');
		}
	}
	
	public function disconnect() {
		$this->sendLine('quit');
		parent::disconnect();
	}
	
	public function __construct($row = null, $cacheObject = null) {
		parent::__construct($row, $cacheObject);
		if(!is_null($cacheObject)) {
			$this->info = $cacheObject->info;
			$this->channels = $cacheObject->channels;
			$this->channelTree = $cacheObject->channelTree;
			$this->clients = $cacheObject->clients;
			$this->clientsToChannels = $cacheObject->clientsToChannels;
			$this->serverQueryClients = $cacheObject->serverQueryClients;
		}
	}
	
	public function getName() {
		$this->info['virtualserver_name'];
	}

	public function getChannelGroup($groupID) {
		if(isset($this->channelGroups[$groupID]))
			return $this->channelGroups[$groupID];
		return false;
	}
	
	public function getChannelGroupName($groupID) {
		if(isset($this->channelGroups[$groupID]))
			return $this->channelGroups[$groupID]['name'];
		return false;
	}

	public function getServerGroups($serverGroups) {
		$result = array();
		$groupIDs = explode(',', $serverGroups);
		foreach($groupIDs as $groupID) {
			if($group = $this->getServerGroup($groupID)) {
				$result[$group['sgid']] = $group;
			}
		}
		return $result;
	}
	
	public function getServerGroup($groupID) {
		if(isset($this->serverGroups[$groupID]))
			return $this->serverGroups[$groupID];
		return false;
	}
	
	public function getServerGroupName($groupID) {
		if(isset($this->serverGroups[$groupID]))
			return $this->serverGroups[$groupID]['name'];
		return false;
	}
	
	public function checkImage($file) {
		return file_exists(WCF_DIR.'images/teamspeak/'.$file.'.png');
	}
	
	public function readData() {
		$this->lastUpdate = TIME_NOW;
		
		// send login information if it is supplied
		if($this->username != '' && $this->password != '') {
			$this->login($this->username, $this->password);
		}
		
		$this->useByPort($this->voicePort);
		
		$this->info = $this->serverInfo();
				
		$result = $this->channelList(true, true, true, true, true, true, true, true);
		$this->channels = $result['list'];
		$this->channelTree = $result['tree'];
		
		$this->serverGroups = $this->serverGroupList();
		
		$this->channelGroups = $this->channelGroupList();

		$clients = $this->clientList(true, true, true, true, true);
		$this->clients = $clients['list'];
		$this->serverQueryClients = $clients['serverQuery'];
		$this->clientsToChannels = $clients['toChannels'];		
	}
	
	public function sendRequest($command, $parameters = null) {
		$originalParameters = $parameters;
		if(!is_null($parameters)) {
			foreach($parameters as $parameter => &$value) {
				if(is_array($value)) {
					foreach($value as &$subValue) {
						$subValue = $parameter.'='.str_replace(self::$parameterSearch, self::$parameterReplace, $subValue);
					}
					$value = implode('|', $value);
				} else {
					$value = $parameter.'='.str_replace(self::$parameterSearch, self::$parameterReplace, $value);
				}
			}
			$parameters = ' '.implode(' ', $parameters);
		} else {
			$parameters = '';
		}
		$this->sendLine($command.$parameters);
		return $this->readResponse($command, $originalParameters);
	}

	protected function parseParameters($parameters) {
		$result = array();
		$parameters = explode(' ', $parameters);
		if(is_array($parameters)) {
			foreach($parameters as $parameter) {
				if(strpos($parameter, '=') !== false) {
					$parameter = explode('=',$parameter);
					$parameterName = array_shift($parameter);
					$parameterValue = implode('=', $parameter);
					$result[$parameterName] = str_replace(self::$parameterReplace, self::$parameterSearch, $parameterValue);
				} else {
					$result[$parameter] = '';
				}
			}
			return $result;
		}
		return null;
	}
	
	protected function parseResponse($response) {
		$result = array('response'=>array(),'error'=>array('id'=>0,'message'=>'','failed_permid'=>0));
		foreach($response as $line) {
			if(self::stringSection($line) == 'error') {
				$line = $this->parseParameters($line);
				$result['error']['id'] = $line['id'];
				$result['error']['message'] = $line['msg'];
				if(isset($line['failed_permid'])) $result['error']['failed_permid'] = $line['failed_permid'];
			} else {
				$lines = explode('|', $line);
				foreach($lines as $line) {
					if(!is_null($line))
						$result['response'][] = $this->parseParameters($line);
				}
			}
		}
		return $result;
	}
	
	protected function channelGetSpacerInfo($name) {
		$result = array('channel_is_spacer'=>false, 'channel_spacer_id' => 0, 'channel_spacer_align' => self::SPACER_ALIGN_LEFT, 'channel_spacer_text' => '');
		if(preg_match("/\[(l|c|r|\*)?spacer([0-9])\]/i", $name, $matches)) {
			$name = preg_replace("/\[(l|c|r|\*)?spacer([0-9])\]/i", '', $name);
			$result['channel_spacer_text'] = $name;
			switch($matches[1]) {
				case 'c':
					$result['channel_spacer_align'] = self::SPACER_ALIGN_CENTER;
					break;
				case 'r':
					$result['channel_spacer_align'] = self::SPACER_ALIGN_RIGHT;
					break;
				case '*':
					$result['channel_spacer_align'] = self::SPACER_ALIGN_REPEAT;
					while(strlen($result['channel_spacer_text']) < 200) {
						$result['channel_spacer_text'] .= $name;
					}
					break;
				default:
					$result['channel_spacer_align'] = self::SPACER_ALIGN_LEFT;
					break;
			}
			$result['channel_spacer_id'] = number_format($matches[2]);
			$result['channel_is_spacer'] = true;			
		}
		return $result;
	}
	
	public function readResponse($command, $parameters) {
		$response = array();
		do {
			$line = $this->readLine();
			$response[] = (CHARSET == 'UTF-8') ? $line : StringUtil::convertEncoding('UTF-8', CHARSET, $line);
		} while(is_string($line) && self::stringSection($line) != 'error');
		return array_merge(array('request'=>array('command'=>$command,'parameters'=>$parameters)), $this->parseResponse($response));
	}
	
	public function login($username, $password) {
		$result = $this->sendRequest('login', array('client_login_name'=>$username,'client_login_password'=>$password));
		if($response['error']['id'] != 0)
			throw new TeamSpeakException($response['error']['message'], $response['error']['id'], $response);
	}

	public function useByServerID($sid) {
		$response = $this->sendRequest('use', array('sid'=>$sid));
		if($response['error']['id'] != 0)
			throw new TeamSpeakException($response['error']['message'], $response['error']['id'], $response);
	}
	
	public function useByPort($port) {
		$response = $this->sendRequest('use', array('port'=>$port));
		if($response['error']['id'] != 0)
			throw new TeamSpeakException($response['error']['message'], $response['error']['id'], $response);
	}
	
	public function serverGroupList() {
		$response = $this->sendRequest('servergrouplist');
		if($response['error']['id'] != 0) {
			throw new TeamSpeakException($response['error']['message'], $response['error']['id'], $response);
		}
		$serverGroups = array();
		foreach($response['response'] as $group) {
			$group['image'] = 'ts3group'.$group['iconid'];
			$group['imageExists'] = $this->checkImage($group['image']);
			$serverGroups[$group['sgid']] = $group;
		}
		return $serverGroups;
	}

	public function serverInfo() {
		$response = $this->sendRequest('serverinfo');
		if($response['error']['id'] != 0)
			throw new TeamSpeakException($response['error']['message'], $response['error']['id'], $response);
		return $response['response'][0];
	}
	
	public function channelGroupList() {
		$response = $this->sendRequest('channelgrouplist');
		if($response['error']['id'] != 0) {
			throw new TeamSpeakException($response['error']['message'], $response['error']['id'], $response);
		}
		$channelGroups = array();
		foreach($response['response'] as $group) {
			$group['image'] = 'ts3group'.$group['iconid'];
			$group['imageExists'] = $this->checkImage($group['image']);
			$channelGroups[$group['cgid']] = $group;
		}
		return $channelGroups;
	}
	
	public function channelList($topic = false, $flags = false, $voice = false, $limits = false) {
		$response = $this->sendRequest('channellist'.($topic?' -topic':'').($flags?' -flags':'').($voice?' -voice':'').($limits?' -limits':''));
		if($response['error']['id'] != 0)
			throw new TeamSpeakException($response['error']['message'], $response['error']['id'], $response);
		$channels = array('list'=>array(),'tree'=>array());
		$i = 0;
		foreach($response['response'] as $channel) {
			$channels['list'][$channel['cid']] = array_merge($channel, $this->channelGetSpacerInfo($channel['channel_name']));
			$channels['tree'][$channel['pid']][$i++] = $channel['cid'];
			ksort($channels['tree'][$channel['pid']]);
		}
		return $channels;
	}
	
	public function clientList($uid = false, $away = false, $voice = false, $times = false, $groups = false, $info = false, $icon = false, $country = false) {
		$response = $this->sendRequest('clientlist'.($uid?' -uid':'').($away?' -away':'').($voice?' -voice':'').($times?' -times':'')
					.($groups?' -groups':'').($info?' -info':'').($icon?' -icon':'').($country?' -country':''));
		if($response['error']['id'] != 0)
			throw new TeamSpeakException($response['error']['message'], $response['error']['id'], $response);
		
		$clients = array('list'=>array(),'toChannels'=>array(),'serverQuery'=>array());
		foreach($response['response'] as $client) {
			if($client['client_type'] == 0) {
				$client['channel_group'] = $this->getChannelGroup($client['client_channel_group_id']);
				$client['servergroups'] = $this->getServerGroups($client['client_servergroups']); 
				$clients['list'][$client['clid']] = $client;
				$clients['toChannels'][$client['cid']][] = $client['clid'];
			} else {
				$clients['serverQuery'][$client['clid']] = $client;
			}
		}
		return $clients;
	}
}