<?php

require_once(WCF_DIR.'lib/util/StringUtil.class.php');


require_once(WCF_DIR.'lib/data/server/AbstractServer.class.php');

/**
 * The TeamSpeak2Server class provides access to a TeamSpeak 2 Server.
 * 
 * @author	Sven Kutzner
 * @copyright	2010 deixu.net
 * @license	GNU Lesser General Public License <http://opensource.org/licenses/lgpl-license.php>
 * @package net.deixu.wcf.teamspeak
 * @subpackage	data
 * @category 	TeamSpeak Viewer
 */
class TeamSpeak2Server extends AbstractServer {
	protected $channels;
	protected $channelTree;
	protected $clients;
	protected $clientsToChannels;

	public function connect() {
		parent::connect();
		if($this->readLine() != '[TS]') {
			throw new SystemException('This is not a TeamSpeak 2 Server.');
		}
	}
	
	public function disconnect() {
		$this->sendLine('quit');
		parent::disconnect();
	}
	
	private static function stripQuotes($s)
	{
		return str_replace('"', '', $s);
	}
	
	public static function getCodec($codec = -1)
	{
		switch($codec)
		{
			case 0: return 'CELP 5.1 Kbit'; break;
			case 1: return 'CELP 6.3 Kbit'; break;
			case 2: return 'CELP 14.8 Kbit'; break;
			case 3: return 'CELP 16.4 Kbit'; break;
			case 4: return 'CELP Windows 5.2 Kbit'; break;
			case 5: return 'Speex 3.4 Kbit'; break;
			case 6: return 'Speex 5.2 Kbit'; break;
			case 7: return 'Speex 7.2 Kbit'; break;
			case 8: return 'Speex 9.3 Kbit'; break;
			case 9: return 'Speex 12.3 Kbit'; break;
			case 10: return 'Speex 16.3 Kbit'; break;
			case 11: return 'Speex 19.5 Kbit'; break;
			case 12: return 'Speex 25.9 Kbit'; break;
			default: return 'Unknown Codec ('.$codec.')'; break;
		}
	}

	public static function userStatus($user)
	{
		if($user['flags'] & 8) return 'away';
		elseif($user['flags'] & 16) return 'mutemicro';
		elseif($user['flags'] & 32) return 'mutespeakers';
		elseif($user['flags'] & 64) return 'record';
		else return 'normal';
		return '';
	}

	public static function getPrivileges($user, $group)
	{
		if($user == 0)
		{
			$result[] = 'U';
		}
		else
		{
			if($user & 4) $result[] = 'R';
			if($user & 1) $result[] = 'SA';
		}
		if($group & 1) $result[] = 'CA';
		if($group & 2) $result[] = 'O';
		if($group & 4) $result[] = 'V';
		if($group & 8) $result[] = 'AO';
		if($group & 16) $result[] = 'AV';

		return $result;
	}
	
	public function getName() {
		return $this->info['server_name'];
	}
	
	public function getChannelTree($parentID = -1) {
		return parent::getChannelTree($parentID);
	}
	
	public function readData() {
		$this->lastUpdate = TIME_NOW;
		
		// select the virtual server
		$result = $this->sendRequest('sel', array($this->voicePort));
		if($result['responseType'] == 'error')
			throw new TeamSpeakException($result['error']['message'], $result['error']['id'], $result);
			
		$result = $this->sendRequest('gi');
		if($result['responseType'] == 'error')
			throw new TeamSpeakException($result['error']['message'], $result['error']['id'], $result);
		$this->info = $result['response'];

		$result = $this->sendRequest('si');
		if($result['responseType'] == 'error')
			throw new TeamSpeakException($result['error']['message'], $result['error']['id'], $result);
		$this->info = array_merge($this->info, $result['response']);
		
		$result = $this->sendRequest('cl');
		if($result['responseType'] == 'error')
			throw new TeamSpeakException($result['error']['message'], $result['error']['id'], $result);
			
		$channels = $result['response'];
		foreach($channels as &$channel) {
			$this->channels[$channel['id']] = $channel;
			$this->channelTree[$channel['parent']][sprintf('[%05s]', $channel['order']).'_'.$channel['name']] = $channel['id'];
			ksort($this->channelTree[$channel['parent']]);
		}

		$result = $this->sendRequest('pl');
		if($result['responseType'] == 'error')
			throw new TeamSpeakException($result['error']['message'], $result['error']['id'], $result);
			
		$this->clients = $result['response'];
		foreach($this->clients as $client) {
			$this->clientsToChannels[$client['c_id']][] = $client['p_id'];
		}		
	}
	
	public function sendRequest($command, $parameters = null) {
		$originalParameters = $parameters;
		if(!is_null($parameters)) {
			$parameters = ' '.implode(' ', $parameters);
		} else {
			$parameters = '';
		}
		$this->sendLine($command.$parameters);
		return $this->readResponse($command, $originalParameters);
	}
	
	public function readResponse($command, $parameters) {
		$response = array();
		do {
			$line = $this->readLine();
			$section = self::stringSection($line);
			$response[] = (StringUtil::isUTF8($line)) ? $line : StringUtil::convertEncoding('ISO-8859-1', CHARSET, $line);
		} while(is_string($line) && $section != 'error' && $section != 'ERROR,' && $section != 'OK');
		return array_merge(array('request'=>array('command'=>$command,'parameters'=>$parameters)), $this->parseResponse($command, $response));
	}
	
	protected function parseResponse($command, $response) {
		$result = array('response'=>array(), 'responseType'=>null);
		switch($command) {
			case 'sel':
				foreach($response as $row) {
					$section = strtolower(self::stringSection($row, 0, ', '));
					$result['responseType'] = $section; 
					$section = self::stringSection($row, 1, ', ');
					$result['error'] = array('id'=>0, 'message'=>$section);
				}
				break;
			case 'cl':
				$header = explode("\t", array_shift($response));
				foreach($response as $row) {
					$section = self::stringSection($row);
					if($section != 'error' && $section != 'OK') {
						$row = array_combine($header, explode("\t", $row));
						$row['id'] = intval($row['id']);
						$row['cid'] = $row['id'];
						$row['codec'] = intval($row['codec']);
						$row['parent'] = intval($row['parent']);
						$row['pid'] = $row['parent']	;
						$row['order'] = intval($row['order']);
						$row['maxusers'] = intval($row['maxusers']);
						$row['flags'] = intval($row['flags']);
						$row['password'] = intval($row['password']);
						$row['name'] = self::stripQuotes($row['name']);
						$row['channel_name'] = $row['name'];
						$row['topic'] = self::stripQuotes($row['topic']);
						$result['response'][$row['id']] = $row;
					} else {
						$result['responseType'] = $section;
						$result['error'] = array('id'=>0, 'message'=>$row);
					}
				}
				break;
			case 'pl':
				$header = explode("\t", array_shift($response));
				foreach($response as $row) {
					$section = self::stringSection($row);
					if($section != 'error' && $section != 'OK') {
						$row = array_combine($header, explode("\t", $row));
						$row['p_id'] = intval($row['p_id']);
						$row['c_id'] = intval($row['c_id']);
						$row['ps'] = intval($row['ps']);
						$row['bs'] = intval($row['bs']);
						$row['pr'] = intval($row['pr']);
						$row['br'] = intval($row['br']);
						$row['pl'] = intval($row['pl']);
						$row['ping'] = intval($row['ping']);
						$row['logintime'] = intval($row['logintime']);
						$row['idletime'] = intval($row['idletime']);
						$row['cprivs'] = intval($row['cprivs']);
						$row['pprivs'] = intval($row['pprivs']);
						$row['privileges'] = self::getPrivileges($row['pprivs'], $row['cprivs']);
						$row['pflags'] = intval($row['pflags']);
						$row['ip'] = str_replace('"','',$row['ip']);
						$row['nick'] = self::stripQuotes($row['nick']);
						$row['loginname'] = self::stripQuotes($row['loginname']);
						$result['response'][$row['p_id']] = $row;
					} else {
						$result['responseType'] = $section;
						$result['error'] = array('id'=>0, 'message'=>$row);
					}
				}
				break;
			case 'si':
			case 'gi':
				foreach($response as $row) {
					$section = self::stringSection($row);
					if($section != 'error' && $section != 'OK') {
						$row = explode('=', $row);
						$key = array_shift($row);
						$row = implode('=', $row);
						$result['response'][$key] = $row;
					} else {
						$result['responseType'] = $section;
						$result['error'] = array('id'=>0, 'message'=>$row);
					}
				}
				break;
		}
		return $result;
	}
}